﻿#region INSTALL SHAREPOINT MODULE

<# Module Name: SharePointPnPPowerShell2013
        Version:     3.9.1905.2
   Repository:  PSGallery #>
# Use the following to locate the module.
Find-Module 'SharePointPnPPowerShell2013'

# Use the following to install the module from the PSGallery.
Install-Module -Name 'SharePointPnPPowerShell2013' -Force

#endregion INSTALL SHAREPOINT MODULE

#region IMPORT HELPERFUNCTION.PSM1

# Import the HelperFunctions.psm1 script module into PowerShell.

<# Use the Get-Module cmdlet to get the folder path to the version
 of the PWPS_DAB module currently in use. #>
$PWPS_DAB_Info = Get-Module -Name PWPS_DAB
$Path = $PWPS_DAB_Info.Path.Substring(0, $PWPS_DAB_Info.Path.LastIndexOf('\'))

# Import HelperFunctions.psm1
Import-Module -Name "$Path\HelperFunctions.psm1" -Force
<# List the available SharePoint functions available within the 
   HelperFunctions script module. #>
Get-Command -Name '*sharepoint*' -Module 'helperfunctions'

#endregion IMPORT HELPERFUNCTION.PSM1

# Get-Help for each function.
# Included with the PWPS_DAB module
Get-Help -Name Get-PWRichProjectProperties -Full 
Get-Help -Name New-PWRichProject -Full 
Get-Help -Name Update-PWFolderNameProps -Full
# Included with the SharePointPnPPowerShell2013 module
Get-Help -Name Connect-PnPOnline -Full 
Get-Help -Name Disconnect-PnPOnline -Full
Get-Help -Name Get-PnPListItem -Full
Get-Help -Name Set-PnPListItem -Full

#region Get Sharepoint Info

# SharePoint Site URL
$Url = "https://bentley.sharepoint.com/sites/SharePointDemo/"
# SharePoint User and Secure Password
$SharePointUser = 'brian.flaherty@bentley.com' #username@email.com'
$SecurePassword = Read-Host -Prompt 'Enter Password:' -AsSecureString
# Splat containing New-Object cmdlet parameter names and values.
$Splat_Credentials = @{
    TypeName = 'PSCredential'
    ArgumentList = ($SharePointUser, $SecurePassword)
}
$Credentials = New-Object @Splat_Credentials

<# Create connection with the specified SharePoint Site.
    Splat containing Connect-PnPOnline cmdlet parameter names and values. #>
$Splat_SharePoint = @{
    Url = $url
    Credentials = $Credentials
}
$SharePointConnection = Connect-PnPOnline @Splat_SharePoint -ReturnConnection -Verbose


# The SharePoint List we want to return data for.
$SPListName = 'ProjectWise_Work_Area_Properties'
# Get the desired data.
$SPItems = Get-PnpListitem -List $SPListName



# Array of properties to get from SharePoint
$WorkAreaPropertyList = "ProjectNo", "Title", 
    "ProjectDescription", "ProjectLocation", "ProjectInProjectWise"

class WorkAreaProperties
{
    [ValidateNotNullOrEmpty()][string]$Number
    [ValidateNotNullOrEmpty()][string]$Name
    [ValidateNotNullOrEmpty()][string]$Description
    [ValidateNotNullOrEmpty()][string]$Location
    [ValidateNotNullOrEmpty()][string]$InProjectWise
    [ValidateNotNullOrEmpty()][int] $SharePointIndex
}

# ArrayList to store data in
$arrayWorkAreaProperties = New-Object System.Collections.ArrayList

<# Loop through all items returned from SharePoint and populate
 the arrayWorkAreaProperties arraylist with the desired data. #>
$index = 0
foreach ($SPItem in $SPItems) {
    <# Populate the wk variable with values from the current SharePoint item.
        Test each to determine if empty or null. #>
           
    foreach ($item in $SPItem.FieldValues) {        
        $wk = [WorkAreaProperties]@{}

        if( -not ([string]::IsNullOrEmpty($item.ProjectNo))) {
            $wk.Number = $item.ProjectNo
        }
        if( -not ([string]::IsNullOrEmpty($item.Title))) { 
            $wk.Name = $item.Title
        }
        if( -not ([string]::IsNullOrEmpty($item.ProjectDescription))) { 
            $wk.Description = $item.ProjectDescription
        }
        if( -not ([string]::IsNullOrEmpty($item.ProjectInProjectWise))) { 
            $wk.Location = $item.ProjectLocation
        }
        if( -not ([string]::IsNullOrEmpty($item.ProjectInProjectWise))) { 
            $wk.InProjectWise = $item.ProjectInProjectWise
        }
        $wk.SharePointIndex = $index
        <# Add WorkArea properties to the ArrayList. 
            Using Out-Null to suppress output from ArrayList.Add method. #>
        $arrayWorkAreaProperties.Add($wk) | Out-Null
    } # end foreach ($item... (Inner Loop)
    $index++
} # end foreach ($SPItem... (Outer Loop)

#endregion Get Sharepoint Info

#region Update Work Area / Project Properties

<# Loop through each item in the array list and determine if
    the specified project exists within ProjectWise. If it does, 
    continue to the next item. If it does not exist, create a 
    new Work Area / Project and update the corresponding
    Work Area / Project properties. 
    Set the InProjectWise item to Yes once a project has been created. #>
foreach ($item in $arrayWorkAreaProperties) {
    Write-Verbose -Message "Processing $($item.Name)" -Verbose
    # Determine if a project exists within ProjectWise using the name property.
    # In my datasource, all projects are contained in a Projects folder.
    $Splat_Project = @{
        FolderPath = "Projects"
        FolderName = $item.Number
    }

    <# Attempt to get the project within ProjectWise. If an object is returned, 
        the project exists. Continue to the next item. #>
    try { 
        $pwProject = Get-PWRichProjects @Splat_Project -JustOne -ErrorAction Stop
        continue
    } catch {
        Write-Warning -Message "Project '$($item.Number)'; Need to create."
    } # end try / catch...
    
    # Populate hash table with Work Area / Project Property names and values.
    $Properties = @{
        PROJECT_WorkAreaNumber = $item.Number
        PROJECT_WorkAreaName = $item.Name
        PROJECT_WorkAreaDescription = $item.Description
        PROJECT_WorkAreaLocation = $item.Location
    }
    
    #region Create new Work Area / Project
    
    <# We will create our new Work Area / Project in the Projects folder. 
        Include Work Area / Project properties.#>
    $Splat_NewWorkArea = @{
        NewFolderPath = "Projects\$($item.Number)"
        ProjectType = 'MyWorkArea'
        StorageArea = 'Storage'
        ProjectProperties = $Properties
    }
    try { 
        $pwNewProject = New-PWRichProject @Splat_NewWorkArea -ErrorAction Stop
    } catch {
        Write-Warning -Message "Error occurred while attempting to create new Work Area / Project. $($Error[0].Exception.Message)"
    }
    
    #endregion Create new Work Area / Project
    
    #region Update the new project description within ProjectWise.
    
    try {
        $Splat_UpdateDescription = @{
            FolderID = $pwNewProject.ProjectID
            NewDescription = $item.Name
        }
        $Result = Update-PWFolderNameProps @Splat_UpdateDescription -ErrorAction Stop
    } catch {
        Write-Warning -Message "Error occurred while attempting to update the Project description. $($Error[0].Exception.Message)"
    }
    
    #endregion Update the new project description within ProjectWise.
    
    #region Set SharePoint ProjectInProjectWise value to Yes (True).        
    
    try {
        $Splat_SetSPItem = @{
            List = $SPListName
            Identity = $SPItems[$item.SharePointIndex]
            Values = @{ProjectInProjectWise = "$true"}
        }
        Set-PnPListItem @Splat_SetSPItem -ErrorAction Stop
    } catch {
        Write-Warning -Message "Error occurred while attempting to set the InProjectWise value within SharePoint. $($Error[0].Exception.Message)"
    }
    
    #endregion Set SharePoint ProjectInProjectWise value to Yes (True).
    
    
} # end foreach ($item...

#endregion Update Work Area / Project Properties

# Disconnect from the SharePoint site.
Disconnect-PnPOnline -Connection $SharePointConnection
